# Modelleisenbahn - Martin Reiche 2026
# neu: Separate Datei für Klassen und Funktionen
import tkinter as tk
import math
import random
from abc import ABC, abstractmethod

# Globale Konstanten 
RADIUS = 180 # Radius der Kurven in Pixel
ANZAHL_KURVENSEGMENTE = 16 # für den Vollkreis
SEGMENT_WINKEL = 360 / ANZAHL_KURVENSEGMENTE
LENGTH = SEGMENT_WINKEL * math.pi / 180 * RADIUS * 1.1
ZUGBREITE = 16
GLEISBREITE = 4
WEICHENINDIKATOR = 8

# Globale Variablen
läuft = False
zeit = 0
t_anim = 0  # zählt von 0 bis ANIMATIONEN_PRO_SIMULATIONSSCHRITT-1
after_id = 1
gleise = dict()
weichen = dict()
züge = dict()

class Vector(): # Vector-Klasse für rechtsdrehendes Koordinatensystem

    def __init__(self, x, y):
        self.x = x
        self.y = y
    
    def __eq__(self, other):
        if math.isclose(self.x, other.x) and math.isclose(self.y, other.y):
            return True
        else:
            return False
        
    def __str__(self):
        return f'x={self.x:0.4g}, y={self.y:0.4g}'

    def __add__(self, other):
        return Vector(self.x + other.x, self.y + other.y)
        
    def __sub__(self, other):
        return Vector(self.x - other.x, self.y - other.y)
    
    def __mul__(self, factor):
        return Vector(self.x * factor, self.y * factor)
        
    def length(self):
        return math.sqrt(self.x**2 + self.y**2)

    def norm(self):
        n = self.length()
        return Vector(self.x / n, self.y / n)
        
    def distance(self, other):
        diff = self - other
        return diff.length()
    
    def rotate(self, alpha):
        x = self.x * math.cos(alpha) + self.y * math.sin(alpha)
        y = self.y * math.cos(alpha) - self.x * math.sin(alpha)
        return Vector(x, y)

    def rotate_deg(self, alpha):
        return self.rotate(alpha / 180 * math.pi)

class Gleis(ABC):
        
    def initialize(self, vorgelagert):
        self.nummer = len(gleise)
        gleise[self.nummer] = self
        vorgelagert.nachgelagert = self
        self.vorgelagert = vorgelagert
        self.startpunkt = vorgelagert.endpunkt
        self.startwinkel = vorgelagert.endwinkel
        self.endwinkel = None # Muss von der Unterklasse definiert werden
        self.nachgelagert = None
        self.befahren = False
        self.tk_handle = 0
        self.tk_handle_zug = 0
        self.tk_handle_gleis = 0
        self.weiche = None

    def registriere_weiche(self, w):
        self.weiche = w
        
    def gib_vorgelagert(self): # Nur für Start
        return self.vorgelagert
    
    def gib_nächstes_gleis(self, kommt_von):
        weiche = self.weiche
        if weiche and weiche.gabel == self:
            return weiche.gib_nächstes_gleis(kommt_von)
                
        if kommt_von == self.vorgelagert:
            return self.nachgelagert
        elif kommt_von == self.nachgelagert:
            return self.vorgelagert
        else:
            raise RuntimeError('Konfigurationsfehler')        
    
    def befahre(self, zug, start=False ):
        if self.befahren:
            raise RuntimeError('Kollision: Dieses Gleis wird schon befahren!')
        else:
            self.befahren = zug
            if start or not ANIMATION:
                self.zeichne(True)
        
    def verlasse(self, zug):
        if self.befahren == zug:
            self.befahren = None
            if not ANIMATION:
                self.zeichne(False)
        elif self.befahren:
            raise RuntimeError('Fehler: Dieses Gleis wird von ' + \
                        'einem anderen Zug befahren')        
        else:
            raise RuntimeError('Fehler: Dieses Gleis kann nicht ' + \
                        'verlassen werden, weil sie nicht befahren war!')

    @abstractmethod
    def zeichne(self, befahre):  # ohne Animation
        ...

    @abstractmethod
    def fülle(self, anteil, rückwärts):    # 
        ...

    @abstractmethod
    def leere(self, anteil, rückwärts):
        ...
        
    @abstractmethod
    def __str__(self):
        ...

class Starter(): # Surrogat für Vorläufer als Starthilfe beim Gleisaufbau  
    def __init__(self, position, winkel):
        self.endpunkt = position
        self.endwinkel = winkel
        
class Gerade(Gleis):
    def __init__(self, vorgelagert, folgende, length=LENGTH):
        super(Gerade, self).initialize(vorgelagert)
        self.endwinkel = self.startwinkel
        self.vektor = Vector(length, 0).rotate_deg(self.endwinkel)
        self.endpunkt = self.startpunkt + self.vektor
        if folgende > 0:
            self.nachgelagert = Gerade(self, folgende-1, length) # Achtung Rekursion!

    def zeichne(self, zeichne):
        linie = [self.startpunkt.x, self.startpunkt.y,
                self.endpunkt.x, self.endpunkt.y]
        
        if zeichne:
            width = ZUGBREITE
            fill = self.befahren.color
        else:
            width = GLEISBREITE
            fill = 'black'
        
        canvas.delete(self.tk_handle)
        self.tk_handle = canvas.create_line(linie, width=width, fill=fill)

    def _berechne_teilstrecken(self, t_anim, rückwärts):
        canvas.delete(self.tk_handle)
        canvas.delete(self.tk_handle_gleis)
        canvas.delete(self.tk_handle_zug)
        
        anteil = (t_anim + 1) / ANIMATIONEN_PRO_SIMULATIONSSCHRITT
        teilvektor = self.vektor * anteil
        
        if rückwärts:
            teilpunkt = self.endpunkt - teilvektor
        else:
            teilpunkt = self.startpunkt + teilvektor

        linie_1 = [self.startpunkt.x, self.startpunkt.y, teilpunkt.x, teilpunkt.y]
        linie_2 = [teilpunkt.x, teilpunkt.y, self.endpunkt.x, self.endpunkt.y]

        if rückwärts:
            linie_1, linie_2 = linie_2, linie_1
            
        return linie_1, linie_2

    def fülle(self, t_anim, color, rückwärts):
        linie_1, linie_2 = self._berechne_teilstrecken(t_anim, rückwärts)

        self.tk_handle_zug = \
            canvas.create_line(linie_1, width=ZUGBREITE, fill=color) 

        self.tk_handle_gleis = \
            canvas.create_line(linie_2, width=GLEISBREITE, fill='black')
        
    def leere(self, t_anim, color, rückwärts):
        linie_1, linie_2 = self._berechne_teilstrecken(t_anim, rückwärts)
            
        self.tk_handle_gleis = \
            canvas.create_line(linie_1, width=GLEISBREITE, fill='black')

        self.tk_handle_zug = \
            canvas.create_line(linie_2, width=ZUGBREITE, fill=color)           

    def __str__(self):
        w = f' Gabel der Weiche {self.weiche.nummer}' if self.weiche else ''
        return 'Gleis ' + str(self.nummer) + ' Geradenstück bei ' \
               + str(self.startpunkt) + ' mit startwinkel ' \
               + str(self.startwinkel) + w
      
        
class Kurve(Gleis):
    def __init__(self, vorgelagert, rechtskurve, folgende, umgekehrt=False):
        super(Kurve, self).initialize(vorgelagert)
        self.rechtskurve = rechtskurve  # boolean
        
        if umgekehrt:
            endwinkel = (vorgelagert.endwinkel + 180) % 360
        else:
            endwinkel = vorgelagert.endwinkel
            
        if rechtskurve:
            self.endwinkel = endwinkel - SEGMENT_WINKEL
        else:
            self.endwinkel = endwinkel + SEGMENT_WINKEL
            
        # Bestimme Mittelpunkt m des Kreises
        s = Vector(1, 0)
        if self.rechtskurve:
            drehwinkel = endwinkel - 90
        else:
            drehwinkel = endwinkel + 90
        s = s.rotate_deg(drehwinkel) * RADIUS
        m = vorgelagert.endpunkt + s
        #canvas.create_rectangle((m.x, m.y, m.x+2, m.y+2))
        xmin = m.x - RADIUS
        ymin = m.y - RADIUS
        xmax = m.x + RADIUS
        ymax = m.y + RADIUS
        self.bbox = xmin, ymin, xmax, ymax
        #canvas.create_rectangle(self.bbox)
        
        # Endpunkt erzeugen
        if self.rechtskurve:
            self.endpunkt = m - s.rotate_deg(-SEGMENT_WINKEL)
        else:
            self.endpunkt = m - s.rotate_deg(SEGMENT_WINKEL)
                
        if umgekehrt:
            self.startwinkel = (vorgelagert.endwinkel + 180) % 360

        # Anschluss-Kurven erzeugen
        if folgende > 0:
            Kurve(self, rechtskurve, folgende-1)  # Achtung Rekursion!
 
    def zeichne(self, zeichne):
        if self.rechtskurve:
            gamma = 90 - SEGMENT_WINKEL + self.startwinkel
        else:
            gamma = 270 + self.startwinkel
            
        if zeichne:
            width = ZUGBREITE
            outline = self.befahren.color
        else:
            width = GLEISBREITE
            outline = 'black'

        canvas.delete(self.tk_handle)
        self.tk_handle = canvas.create_arc(self.bbox, start=gamma,
                         extent=SEGMENT_WINKEL, style=tk.ARC, width=width,
                         outline=outline)

    def fülle(self, t_anim, color, rückwärts):
        canvas.delete(self.tk_handle)
        canvas.delete(self.tk_handle_gleis)
        canvas.delete(self.tk_handle_zug)
        
        anteil = (t_anim + 1) / ANIMATIONEN_PRO_SIMULATIONSSCHRITT
        
        if self.rechtskurve:
            anfang = 90 + self.startwinkel
            ende   = anfang - SEGMENT_WINKEL
        else:
            anfang = 270 + self.startwinkel # = gamma
            ende   = anfang + SEGMENT_WINKEL

        if rückwärts:
            start_1 = ende
            winkel_1 = (anfang - ende) * anteil
            start_2 = ende + winkel_1
            winkel_2 = (anfang - ende) * (1 - anteil)
        else:
            start_1 = anfang
            winkel_1 = (ende - anfang) * anteil
            start_2 = anfang + winkel_1
            winkel_2 = (ende - anfang) * (1 - anteil)
            
        self.tk_handle_zug = canvas.create_arc(self.bbox, start=start_1,
                 extent=winkel_1, style=tk.ARC, width=ZUGBREITE,
                 outline=color)
        
        self.tk_handle_gleis = canvas.create_arc(self.bbox, start=start_2,
                 extent=winkel_2, style=tk.ARC, width=GLEISBREITE,
                 outline='black')
            
    def leere(self, t_anim, color, rückwärts):
        canvas.delete(self.tk_handle)
        canvas.delete(self.tk_handle_gleis)
        canvas.delete(self.tk_handle_zug)

        anteil = (t_anim + 1) / ANIMATIONEN_PRO_SIMULATIONSSCHRITT
        
        if self.rechtskurve:
            anfang = 90 + self.startwinkel
            ende   = anfang - SEGMENT_WINKEL
        else:
            anfang = 270 + self.startwinkel # = gamma
            ende   = anfang + SEGMENT_WINKEL

        if rückwärts:
            start_1 = ende
            winkel_1 = (anfang - ende) * anteil
            start_2 = ende + winkel_1
            winkel_2 = (anfang - ende) * (1 - anteil)
        else:
            start_1 = anfang
            winkel_1 = (ende - anfang) * anteil
            start_2 = anfang + winkel_1
            winkel_2 = (ende - anfang) * (1 - anteil)
            
        self.tk_handle_zug = canvas.create_arc(self.bbox, start=start_2,
                 extent=winkel_2, style=tk.ARC, width=ZUGBREITE,
                 outline=color)
        
        self.tk_handle_gleis = canvas.create_arc(self.bbox, start=start_1,
                 extent=winkel_1, style=tk.ARC, width=GLEISBREITE,
                 outline='black')
        
    def __str__(self):
        r = 'Rechtskurve' if self.rechtskurve else 'Linkskurve'
        w = f' Gabel der Weiche {self.weiche.nummer}' if self.weiche else ''
        return 'Gleis ' + str(self.nummer) + ' ' + r + \
               ' bei ' + str(self.startpunkt) + \
               ' mit Startwinkel ' + str(self.startwinkel) + ' ' + w
    
class Weiche():
    def __init__(self, e, g, l, r):
        self.nummer = len(weichen)
        weichen[self.nummer] = self
        self.eingang = gleise[e]
        self.gabel   = gleise[g]
        self.linke   = gleise[l]
        self.rechte  = gleise[r]
        self.nach_links = True 
        self.gabel.registriere_weiche(self)
#         print('Weiche #', self.nummer)
#         print('Eingang start =', self.eingang.startpunkt, \
#               'Eingang ende  =', self.eingang.endpunkt)
#         print('Gabel start   =', self.gabel.startpunkt, \
#               'Gabel ende    =', self.gabel.endpunkt)
        self.bestimme_bbox_indikator()
        self.zeichne_indikator()
        
    def schalte(self, nach_links):
        if nach_links:
           self.nach_links = True
           # print(self)
        else:
           self.nach_links = False
           # print(self)
        self.zeichne_indikator()

    def gib_nächstes_gleis(self, kommt_von):
        if kommt_von == self.eingang:
            if self.nach_links:
                return self.linke
            else:
                return self.rechte
        else:
            return self.eingang               
        
    def __str__(self):
        richtung = 'links' if self.nach_links else 'rechts'
        return f'Weiche #{self.nummer} mit gabel={self.gabel.nummer}'  \
               f', linke={self.linke.nummer}'  \
               f', rechte={self.rechte.nummer}, steuert nach {richtung}'
    
    def bestimme_bbox_indikator(self):
        ''' Berechnet die tkinter bboxen für die Richtungsanzeige '''
        # Berechne Richtungsvektor der Weiche
        if self.gabel.startpunkt == self.eingang.endpunkt \
           or self.gabel.startpunkt == self.eingang.startpunkt:
            s = self.gabel.startpunkt
            r = self.gabel.endpunkt - self.gabel.startpunkt
        else:
            s = self.gabel.endpunkt 
            r = self.gabel.startpunkt - self.gabel.endpunkt

        # Berechne Ort der Indikatoren
        closer = 0.2       
        p_links  = s + r + r.rotate_deg(90) * closer
        p_rechts = s + r + r.rotate_deg(-90) * closer

        # Berechne die bboxen
        W = WEICHENINDIKATOR / 2
        pl = p_links
        self.bbox_links = (pl.x-W, pl.y-W, pl.x+W, pl.y+W)
        self.tk_handle_links = 0
        pr = p_rechts
        self.bbox_rechts = (pr.x-W, pr.y-W, pr.x+W, pr.y+W)
        self.tk_handle_rechts = 0

    def zeichne_indikator(self):
        if self.nach_links:
            canvas.delete(self.tk_handle_rechts)
            self.tk_handle_links = \
            canvas.create_oval(self.bbox_links, fill='black')
        else:
            canvas.delete(self.tk_handle_links)
            self.tk_handle_rechts = \
            canvas.create_oval(self.bbox_rechts, fill='black')            
        
        
class Zug():
    WARTEZEIT = 3

    def __init__(self, color, länge, start):
        self.nummer = len(züge)
        züge[self.nummer] = self        
        self.länge = länge
        self.belegte_gleise = []
        self.color = color
        self.wartezeit = 0
        self.kehr_um = False

        # Kopf des Zuges    
        self.aktuelles_gleis = gleise[start]
        self.letztes_gleis = self.aktuelles_gleis.gib_vorgelagert()
        self.aktuelles_gleis.befahre(self, True)

        # Körper des Zuges
        gleis_nummer = start
        for g in range(länge):
            self.belegte_gleise.append(gleis_nummer)
            gleis = gleise[gleis_nummer]
            gleis.befahren = self
            gleis.zeichne(True)
            gleis = gleise[gleis_nummer].gib_vorgelagert()
            gleis_nummer = gleis.nummer
        
        self.zuletzt_verlassenes_gleis = None
        
    def update(self):
        aktuelles_gleis = \
            self.aktuelles_gleis.gib_nächstes_gleis(self.letztes_gleis)
        
        if self.wartezeit:
            self.wartezeit -= 1
            if self.wartezeit == 0:
                if aktuelles_gleis.befahren:
                    print(self.color, 'fährt jetzt zurück')
                    self.wechsle_richtung()
            else:
                print(self.color, 'wartet noch', self.wartezeit)
                return
                
        if aktuelles_gleis.befahren:
            self.wartezeit = self.WARTEZEIT
            print(self.color, 'wartet jetzt', self.wartezeit)
            return

        gleise[self.belegte_gleise[-1]].verlasse(self)
        self.zuletzt_verlassenes_gleis = gleise[self.belegte_gleise.pop(-1)]
        self.belegte_gleise.insert(0, aktuelles_gleis.nummer)
        aktuelles_gleis.befahre(self)
        self.letztes_gleis = self.aktuelles_gleis
        self.aktuelles_gleis = aktuelles_gleis
        
    def wechsle_richtung(self):
        
        # Invertiere die Liste der self.belegte_gleise
        self.belegte_gleise.reverse()
        # Aktuelles Gleis ist das jetzt erste Element in belegte Gleise
        self.aktuelles_gleis = gleise[self.belegte_gleise[0]]
        # Wir brauchen ein zuletzt verlassenes Gleis als member!
        if len(self.belegte_gleise) == 1:
            self.letztes_gleis = self.aktuelles_gleis. \
                                  gib_nächstes_gleis(self.letztes_gleis)
        else:
            self.letztes_gleis = gleise[self.belegte_gleise[1]]
        
    def animiere(self, t_anim):
        
        if self.wartezeit:
            return

        # Leere verlassenes Gleis
        if len(self.belegte_gleise) == 1:
            letztes_gleis = self.letztes_gleis
            aktuelles_gleis = self.aktuelles_gleis
        else:
            letztes_gleis = self.zuletzt_verlassenes_gleis
            aktuelles_gleis = gleise[self.belegte_gleise[-1]]
            
        if letztes_gleis.endpunkt.distance(
           aktuelles_gleis.startpunkt) < 10 or \
           letztes_gleis.endpunkt.distance(
           aktuelles_gleis.endpunkt) < 10:  # Ungenauigkeit im Gleisbild
            rückwärts = False
        else:
            rückwärts = True        
        letztes_gleis.leere(t_anim, self.color, rückwärts)
        
        # Fülle befahrenes Gleis
        if self.letztes_gleis.endpunkt.distance(
           self.aktuelles_gleis.startpunkt) < 10 or \
           self.letztes_gleis.startpunkt.distance(
           self.aktuelles_gleis.startpunkt) < 10:  # Ungenauigkeit im Gleisbild
            rückwärts = False
        else:
            rückwärts = True
        self.aktuelles_gleis.fülle(t_anim, self.color, rückwärts)

    def __str__(self):
        return f'Zug #{self.nummer}: Farbe={self.color}, ' \
               f'Länge={self.länge}, Kopf steht auf Gleis ' \
               f'#{self.aktuelles_gleis.nummer}'
        
def zeige_gleise():
    # Zeichne den Gleisplan
    font = 'Arial', 12
    for nummer, s in gleise.items():
        if isinstance(s, Gerade):
            x = (s.startpunkt.x + s.endpunkt.x) / 2 +10
            y = (s.startpunkt.y + s.endpunkt.y) / 2
        else:
            x = (s.startpunkt.x + s.endpunkt.x) / 2 -10
            y = (s.startpunkt.y + s.endpunkt.y) / 2            
        canvas.create_text(x, y, font=font, text=str(nummer), fill='yellow')
        
    # Markiere die Startpunkte der Gleise
    for nummer, s in gleise.items():
        line = [s.startpunkt.x-2, s.startpunkt.y-2,
                s.startpunkt.x+2, s.startpunkt.y+2]
        canvas.create_rectangle(line, width=1, fill='yellow')

def neuestes_gleis():
    return gleise[len(gleise) - 1]




