'''
Ray Tracer
Martin Reiche
January 2019
'''

import tkinter as tk
import math
import sys


# Geometric classes
class Vector(object):
    def __init__(self, x, y, z):
        self.x = x
        self.y = y
        self.z = z

    def __add__(self, other):
        ret_val = Vector(self.x + other.x, self.y + other.y, self.z + other.z)
        return ret_val

    def __sub__(self, other):
        ret_val = Vector(self.x - other.x, self.y - other.y, self.z - other.z)
        return ret_val

    def __mul__(self, other):
        return self.x * other.x + self.y * other.y + self.z * other.z

    def square(self):
        return self.x ** 2 + self.y ** 2 + self.z ** 2

    def length(self):
        return math.sqrt(self.square())

    def unit(self):
        length = self.length()
        return Vector(self.x / length, self.y / length, self.z / length)

    def scale(self, factor):
        return Vector(self.x * factor, self.y * factor, self.z * factor)

    def __str__(self):
        return 'x=' + f'{self.x:0.3f}' \
                + ', y=' + f'{self.y:0.3f}' \
                + ', z=' + f'{self.z:0.3f}' 

class Ray(object):

    def __init__(self, origin, direction):
        self.origin = origin
        self.direction = direction
        self.reflections = 0

    def perpendicular(self, point):
        ''' Fällt das Lot von point auf diesen Strahl.
        Rückgabe:
            lbd  - (lambda) Entfernung von self.origin zum Fußpunkt p des Lotes
            perp - Lotvektor von point nach p '''
        lbd = self.direction * (point - self.origin) / self.direction.square()
        p = self.origin + self.direction.scale(lbd)
        perp = p - point
        return lbd, perp

    def reflected(self):
        self.reflections += 1

    def __str__(self):
        retval = 'origin: ' + self.origin.__str__() + '\n' \
                +'direction: ' + self.direction.__str__() + '\n'
        return retval
    
class Sphere(object):
    def __init__(self, center, radius):
        self.center = center    # vector
        self.radius = radius    # float

    def reflex(self, ray):
        # returns true upon reflection, ray is modified
        # returns false if ray misses sphere, ray is unmodified

        # Step 1: check for intersection
        # drop perpendicular from center to ray and check length
        lbd, perp = ray.perpendicular(self.center)
        if lbd < 0 or perp.length() > self.radius:
            return False

        # Step 2: Calculate reflected ray (= modify original ray)
        # Calculate intersection point of ray

        delta = math.sqrt(self.radius**2 - perp.square())
        D = self.center + perp - ray.direction.unit().scale(delta)
        
        # Calculate normal ray
        n = Ray(D, D - self.center)
        
        # Drop perpendicular from ray origin to normal ray
        _, z = n.perpendicular(ray.origin)       
        B = ray.origin + z.scale(2.0)
        ray.origin = D
        ray.direction = B - D
        ray.reflected()
        return True

# Test functions begin
def error(name):
    print("Test", name, "failed, exit!")
    sys.exit()


def tests():
    u = Vector(1, 2, 3)
    v = Vector(11, 22, 33)

    w = u + v
    if w.x != 12 or w.y != 24 or w.z != 36:
        error("__add__")

    w = v - u
    if w.x != 10 or w.y != 20 or w.z != 30:
        error("__sub__")

    w = u * v
    if w != 154:
        error("__mul__")

    w = u.square()
    if w != 14:
        error("square")

    w = u.length()
    if w != math.sqrt(14):
        error("length")

    w = u.unit()
    ppd = u.length()
    if w.x != 1 / ppd or w.y != 2 / ppd or w.z != 3 / ppd:
        error("length")

    w = u.scale(3)
    if w.x != 3 or w.y != 6 or w.z != 9:
        error("scale")

    p1 = Vector(3, 1, 2)
    p2 = Vector(2, 2, 3)
    r = Ray(p1, p2 - p1)
    p3 = Vector(2, 1.5, 2)
    l, ppd = r.perpendicular(p3)
    if l != 0.5:
        error("perpendicular1")
    if ppd.x != 0.5 or ppd.y != 0 or ppd.z != 0.5:
        error("perpendicular1")

    p1 = Vector(3, 2, 1)
    p2 = Vector(2, 3, 2)
    r = Ray(p1, p2 - p1)
    p3 = Vector(2, 2, 1.5)
    l, ppd = r.perpendicular(p3)
    if l != 0.5:
        error("perpendicular2")
    if ppd.x != 0.5 or ppd.y != 0.5 or ppd.z != 0.0:
        error("perpendicular2")

    print("All tests passed")
# Test functions end

# Scene setup
imageX = 0  # Lower left corner
imageY = 1
imageWidth = 4
imageHeight = 3
eye = Vector(2.0, 4.5, -8.0)

resolution = 100    # number of pixels per unit 200
nx = imageWidth * resolution
ny = imageHeight * resolution

balls = [Sphere(Vector(0.8, 1.7, 6.5), 1.7),
         Sphere(Vector(3.5, 1.2, 3.5), 1.0),
         Sphere(Vector(0.5, 0.5, 3.0), 0.5)]

def is_shadow(direction, ball):
    l, p = direction.perpendicular(ball.center)
    if l > 0 and p.length() < ball.radius:
        return True
    else:
        return False

def background(ray):
    # The colors of the sky...
    if ray.direction.y >= 0:
        up = Vector(0, 1, 0)
        cos_a = (ray.direction * up) / (up.length() * ray.direction.length())
        alpha = math.acos(cos_a) * 2 / math.pi
        red = 80 + 150 * alpha
        green = 100 + 100 * (1 - alpha)
        blue = 128 + 126 * (1 - alpha)
        return "#%02x%02x%02x" % (int(red), int(green), int(blue))

    # Calculate intersection with plane
    t = - ray.origin.y / ray.direction.y
    z = ray.origin.z + t * ray.direction.z
    x = ray.origin.x + t * ray.direction.x

    # Determine shadow
    origin = Vector(x, 0.0, z)
    direction = Vector(0.2, 1, 0.2)
    v = Ray(origin, direction)
    shadow = False
    for ball in balls:
        shadow |= is_shadow(v, ball)

    # Dim reflected light
    dim = math.pow(0.7, ray.reflections)
    
    # Appy checkerboard
    # Avoid modulo of negative numbers by adding 100
    if (int(x + 100) + int(z + 100)) % 2 == 0:
        if shadow:
            dim = int(128 * dim)
        else:
            dim = int(255 * dim)
    else:
        if shadow:
            dim = 0
        else:
            dim = int(30 * dim)
    return "#%02x%02x%02x" % (dim, dim, dim)

def draw():
    for x in range(nx):
        for y in range(ny):
            point = Vector(imageX + imageWidth * x / nx, imageY + imageHeight * y / ny, 0)
            direction = point - eye
            r = Ray(eye, direction)
            while True:
                any_hit = False
                for ball in balls:
                    any_hit |= ball.reflex(r)
                if not any_hit:
                    break
            color = background(r)
            image.put(color, (x, ny - y))
        gui.update()

# init GUI
gui = tk.Tk()
gui.title('RayTracer')
tests()
canvas = tk.Canvas(gui, width=nx, height=ny, bg="gray")
canvas.pack()
image = tk.PhotoImage(width=nx, height=ny)
canvas.create_image((nx/2, ny/2), image=image, state="normal")
draw()
gui.mainloop()


